﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.Configuration;
using System.ComponentModel;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.EFR;
using VeteransAffairs.Registries.BusinessManager.Utilities;

/// <summary>
/// Summary description for Helpers
/// </summary>
public static class Helpers
{
    private static string[] _countries = new string[] { 
         "Afghanistan", "Albania", "Algeria", "American Samoa", "Andorra", 
         "Angola", "Anguilla", "Antarctica", "Antigua And Barbuda", "Argentina", 
         "Armenia", "Aruba", "Australia", "Austria", "Azerbaijan",
		   "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus",
		   "Belgium", "Belize", "Benin", "Bermuda", "Bhutan",
		   "Bolivia", "Bosnia Hercegovina", "Botswana", "Bouvet Island", "Brazil",
		   "Brunei Darussalam", "Bulgaria", "Burkina Faso", "Burundi", "Byelorussian SSR",
		   "Cambodia", "Cameroon", "Canada", "Cape Verde", "Cayman Islands",
		   "Central African Republic", "Chad", "Chile", "China", "Christmas Island",
		    "Colombia", "Comoros", "Congo", "Cook Islands",
		   "Costa Rica", "Cote D'Ivoire", "Croatia", "Cuba", "Cyprus",
		   "Czech Republic", "Czechoslovakia", "Denmark", "Djibouti", "Dominica",
		   "Dominican Republic", "East Timor", "Ecuador", "Egypt", "El Salvador",
		   "England", "Equatorial Guinea", "Eritrea", "Estonia", "Ethiopia",
		   "Falkland Islands", "Faroe Islands", "Fiji", "Finland", "France",
		   "Gabon", "Gambia", "Georgia", "Germany", "Ghana",
		   "Gibraltar", "Great Britain", "Greece", "Greenland", "Grenada",
		   "Guadeloupe", "Guam", "Guatemela", "Guernsey", "Guiana",
		   "Guinea", "Guinea-Bissau", "Guyana", "Haiti", "Heard Islands",
		   "Honduras", "Hong Kong", "Hungary", "Iceland", "India",
		   "Indonesia", "Iran", "Iraq", "Ireland", "Isle Of Man",
		   "Israel", "Italy", "Jamaica", "Japan", "Jersey",
		   "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Korea, South",
		   "Korea, North", "Kuwait", "Kyrgyzstan", "Lao People's Dem. Rep.", "Latvia",
		   "Lebanon", "Lesotho", "Liberia", "Libya", "Liechtenstein",
		   "Lithuania", "Luxembourg", "Macau", "Macedonia", "Madagascar",
		   "Malawi", "Malaysia", "Maldives", "Mali", "Malta",
		   "Mariana Islands", "Marshall Islands", "Martinique", "Mauritania", "Mauritius",
		   "Mayotte", "Mexico", "Micronesia", "Moldova", "Monaco",
		   "Mongolia", "Montserrat", "Morocco", "Mozambique", "Myanmar",
		   "Namibia", "Nauru", "Nepal", "Netherlands", "Netherlands Antilles",
		   "Neutral Zone", "New Caledonia", "New Zealand", "Nicaragua", "Niger",
		   "Nigeria", "Niue", "Norfolk Island", "Northern Ireland", "Norway",
		   "Oman", "Pakistan", "Palau", "Panama", "Papua New Guinea",
		   "Paraguay", "Peru", "Philippines", "Pitcairn", "Poland",
		   "Polynesia", "Portugal", "Puerto Rico", "Qatar", "Reunion",
		   "Romania", "Russian Federation", "Rwanda", "Saint Helena", "Saint Kitts",
		   "Saint Lucia", "Saint Pierre", "Saint Vincent", "Samoa", "San Marino",
		   "Sao Tome and Principe", "Saudi Arabia", "Scotland", "Senegal", "Seychelles",
		   "Sierra Leone", "Singapore", "Slovakia", "Slovenia", "Solomon Islands",
		   "Somalia", "South Africa", "South Georgia", "Spain", "Sri Lanka",
		   "Sudan", "Suriname", "Svalbard", "Swaziland", "Sweden",
		   "Switzerland", "Syrian Arab Republic", "Taiwan", "Tajikista", "Tanzania",
		   "Thailand", "Togo", "Tokelau", "Tonga", "Trinidad and Tobago",
		   "Tunisia", "Turkey", "Turkmenistan", "Turks and Caicos Islands", "Tuvalu",
		   "Uganda", "Ukraine", "United Arab Emirates", "United Kingdom", "United States",
		   "Uruguay", "Uzbekistan", "Vanuatu", "Vatican City State", "Venezuela",
		   "Vietnam", "Virgin Islands", "Wales", "Western Sahara", "Yemen",
		   "Yugoslavia", "Zaire", "Zambia", "Zimbabwe"};


    /// <summary>
    /// Create three new change history records when unaccepting a lab result
    /// </summary>
    /// <param name="workCaseId"></param>
    /// <param name="labActivityId"></param>
    /// <param name="interpreationActivityId"></param>
    private static void CreateChangeHistoryRecordsWhenUnaccept(
        int workCaseId, int labActivityId, int interpreationActivityId)
    {
        LABRESULT_CHANGE_TRACKING changeHistory = null;
        string url = string.Empty;

        // Provider letter archived
        url = string.Format("../LabResultsInterpretationLetters.aspx?activityId={0}&intActivityId={1}&RptType={2}&CaseId={3}", labActivityId, interpreationActivityId, "1", workCaseId);
        changeHistory = CreateNewChangeHistoryObject(workCaseId, EFRSharedConstant.LabResultsChangeHistoryProviderLetterArchived, url);
        CreatOneChangeHistoryRecord(changeHistory);

        // Patient letter archived
        url = string.Format("../LabResultsInterpretationLetters.aspx?activityId={0}&intActivityId={1}&RptType={2}&CaseId={3}", labActivityId, interpreationActivityId, "0", workCaseId);
        changeHistory = CreateNewChangeHistoryObject(workCaseId, EFRSharedConstant.LabResultsChangeHistoryPatientLetterArchived, url);
        CreatOneChangeHistoryRecord(changeHistory);

        // Unaccept contact log created
        changeHistory = CreateNewChangeHistoryObject(workCaseId, EFRSharedConstant.LabResultsChangeHistoryContactLogCreated, string.Empty);
        CreatOneChangeHistoryRecord(changeHistory);
    }

    /// <summary>
    /// Create a new change history object
    /// </summary>
    /// <param name="workCaseId"></param>
    /// <param name="descriptionText"></param>
    /// <param name="navigationUrl"></param>
    /// <returns></returns>
    private static LABRESULT_CHANGE_TRACKING CreateNewChangeHistoryObject(
        int workCaseId, string descriptionText, string navigationUrl)
    {
        LABRESULT_CHANGE_TRACKING changeHistory = new LABRESULT_CHANGE_TRACKING();

        changeHistory.WKF_CASE_ID = workCaseId;
        changeHistory.CHANGE_DATE = DateTime.Now;
        changeHistory.CHANGE_DESCRIPTION = descriptionText;
        if (!string.IsNullOrEmpty(navigationUrl))
        {
            changeHistory.NAVIGATION_URL = navigationUrl;
        }

        return changeHistory;
    }

    /// <summary>
    /// Insert a new record to LABRESULT_CHANGE_TRACKING table
    /// </summary>
    /// <param name="changeTracking"></param>
    private static void CreatOneChangeHistoryRecord(LABRESULT_CHANGE_TRACKING changeTracking)
    {
        LabResultsTrackingManager manager = new LabResultsTrackingManager();
        changeTracking.SetAsChangeTrackingRoot();
        changeTracking.SetAsInsertOnSubmit();
        manager.Update(changeTracking);
    }
    /// <summary>
    /// Unaccept a lab results
    /// </summary>
    /// <param name="labResultsActivity"></param>
    public static void UnacceptLabResults(WKF_CASE_ACTIVITY labResultsActivity, 
                                          WKF_CASE workCase)
    {
        WorkflowCaseManager wfcManager = new WorkflowCaseManager();
        LabResultsManager labManager = new LabResultsManager();
        bool changeToInprocess = true;

        WKF_CASE_ACTIVITY letterActivity =
            GetInterpretationLetterWorkcaseActivity(labResultsActivity.WKF_CASE_ID,
            (int)WorkflowCaseManager.WorkFlowCaseActivityType.InterpretationDiagnosis);

        if (letterActivity != null)
        {
            // The letter is in process 
            // Delete the letter record and update the activity record to New as status
            if (letterActivity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.InProcess)
            {
                //Delete the interpretation letter record 
                Helpers.DeleteInterpreationLetter(letterActivity.WKF_CASE_ACTIVITY_ID);

                //Delete interpretation letter activity record
                wfcManager.DeleteWorkCaseActivity(letterActivity.WKF_CASE_ACTIVITY_ID);


            }
            // The letter is completed
            else if (letterActivity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.Interpreted)
            {
                changeToInprocess = false;

                // Create a new contact log
                ContactLogManager.ContactLog lastPrintedContact = Helpers.GetLastPrintedContactLog((int)workCase.REFERRAL_ID);
                if (lastPrintedContact != null)
                {
                    string contactMessage = "Lab results were unaccepted for an Interpretation Letter that was printed by "
                        + lastPrintedContact.ContactInitiator + " on "
                        + lastPrintedContact.ContactDate + ".";

                    CreateNewContactLog((int)workCase.PATIENT_ID,
                        (int)workCase.REFERRAL_ID, EFRSharedConstant.ContactLogReasonUnaccepted, contactMessage, EFRSharedConstant.ContactLogMethodSystem);
                }

                // Create a new copy of the current lab results activity 
                labManager.CreateNewCopyOfCurrentLabResults(labResultsActivity.WKF_CASE_ACTIVITY_ID);

                // Archive the current lab results activity 
                wfcManager.UpdateActivityStatus(labResultsActivity.WKF_CASE_ACTIVITY_ID,
                    (int)WorkflowCaseActivityStatus.Accepted, true);

                //Archive the interpreted letter 
                wfcManager.UpdateActivityStatus(letterActivity.WKF_CASE_ACTIVITY_ID,
                    (int)WorkflowCaseActivityStatus.Interpreted, true);

                // Create change history records
                CreateChangeHistoryRecordsWhenUnaccept(labResultsActivity.WKF_CASE_ID,
                    labResultsActivity.WKF_CASE_ACTIVITY_ID,
                    letterActivity.WKF_CASE_ACTIVITY_ID);
            }
        }
        
        if ( changeToInprocess )
        {
            //Change the activity status to In Process
            wfcManager.UpdateActivityStatus(labResultsActivity.WKF_CASE_ACTIVITY_ID,
    (int)WorkflowCaseActivityStatus.InProcess, false);
        }

   }

    /// <summary>
    /// Create a new contact log
    /// </summary>
    /// <param name="patient"></param>
    /// <param name="referralId"></param>
    /// <param name="reason"></param>
    /// <param name="contactMessage"></param>
    /// <param name="contractMethod"></param>
    public static void CreateNewContactLog(int patientId, int referralId,
        string reason, string contactMessage, string contactMethod)
    {
        //Get Patient
        PatientManager patientManager = new PatientManager();
        PATIENT patient = patientManager.SelectByPatientID(patientId);

        //Populate new contact log object
        CONTACT_LOG contact = new CONTACT_LOG();

        contact.REFERRAL_ID = referralId;
        contact.CONTACT_FIRST_NAME = patient.FIRST_NAME;
        contact.CONTACT_LAST_NAME = patient.LAST_NAME;
        contact.CONTACT_DATE = DateTime.Now;
        contact.CONTACT_METHOD_TEXT = contactMethod;
        contact.CONTACT_EMAIL = null;
        contact.CONTACT_ADDR1 = patient.ADDRESS_LINE1;
        contact.CONTACT_ADDR2 = patient.ADDRESS_LINE2;
        contact.CONTACT_ADDR3 = patient.ADDRESS_LINE3;
        contact.CONTACT_CITY = patient.CITY;
        contact.CONTACT_STATE = patient.STATE;
        contact.CONTACT_POSTAL = patient.POSTAL_CODE;
        contact.CONTACT_ZIPPLUS4 = patient.ZIP_PLUS_4;
        contact.CONTACT_COUNTRY = patient.COUNTRY;
        contact.CONTACT_REASON_TEXT = reason;
        contact.TELEMEDICINE_PATIENT_FLAG = false;
        contact.TELEMEDICINE_PROVIDER_FLAG = false;
        contact.INPATIENT_EVAL_RECOM_FLAG = false;
        contact.CONTACT_MESSAGE_TEXT = contactMessage;

        contact.SetAsChangeTrackingRoot();
        contact.SetAsInsertOnSubmit();

        ContactLogManager manager = new ContactLogManager();
        manager.Update(contact);
    }

    /// <summary>
    /// Get the last printed contact log record
    /// </summary>
    /// <param name="referralId"></param>
    /// <returns></returns>
    public static ContactLogManager.ContactLog GetLastPrintedContactLog(int referralId)
    {
        ContactLogManager.ContactLog contactLog = null;

        ContactLogManager manager = new ContactLogManager();
        List<ContactLogManager.ContactLog> contacts = manager.SelectByID(referralId).ToList();
        foreach (ContactLogManager.ContactLog contact in contacts)
        {
            if (contact.ContactReason == EFRSharedConstant.ContactLogReasonPrinted)
            {
                contactLog = contact;
                break;
            }
        }

        return contactLog;
    }

    /// <summary>
    /// Get a string from an object
    /// </summary>
    /// <param name="sessionVariable"></param>
    /// <returns></returns>
    public static string GetStringFromObject(object sessionVariable)
    {
        string result = string.Empty;

        if (sessionVariable != null)
        {
            result = sessionVariable.ToString();
        }

        return result;
    }

    /// <summary>
    /// Delete an interpretation letter by work flow case acivity ID
    /// </summary>
    /// <param name="workcaseActivityId"></param>
    /// <returns></returns>
    public static int DeleteInterpreationLetter(int workcaseActivityId)
    {
        LabResultsIntrprtnManager manager = new LabResultsIntrprtnManager();
        return manager.DeleteInterpretationLetter(workcaseActivityId);
    }


    /// <summary>
    /// Get the work flow case activity record for interpretation letter
    /// </summary>
    /// <param name="workcaseId"></param>
    /// <param name="activityTypeId"></param>
    /// <returns></returns>
    public static WKF_CASE_ACTIVITY GetInterpretationLetterWorkcaseActivity(int workcaseId, int activityTypeId)
    {
        WorkflowCaseManager wfcManager = new WorkflowCaseManager();
        return wfcManager.SelectByCaseIdAndActivityTypeId(workcaseId,
            activityTypeId);
    }

    /// <summary>
    /// Unaccept lab results functionality should only be accessible to 
    /// the "TEFSC Coordinator (superuser)" role.   
    /// </summary>
    /// <param name="activityStatusId"></param>
    public static void UnacceptLabResults(int activityStatusId, int workcaseId,
        Button button, UserAccountManager currentUser)
    {
        if (currentUser.GetRoles().Contains((int)RoleCode.EFR_TEFSC_COORDINATOR))
        {
            button.Enabled = true;
            if (activityStatusId == (int)WorkflowCaseActivityStatus.Accepted)
            {
                button.Text = "Unaccept";

                WKF_CASE_ACTIVITY activity 
                    = GetInterpretationLetterWorkcaseActivity(workcaseId,
                    (int)WorkflowCaseManager.WorkFlowCaseActivityType.InterpretationDiagnosis);

                string message1 = "Are you sure that you want to Unaccept this lab result?";
                string message2 = string.Empty;
                string message3 = string.Empty;
                string message4 = string.Empty;
                string message5 = string.Empty;
                string message6 = string.Empty;
                if (activity != null)
                {
                    if (activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.InProcess)
                    {
                        message2 = "Intepretation letters have been started but not completed.";
                        message3 = "These letters will be deleted.";
                    }
                    else if (activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.Interpreted)
                    {
                        message2 = "Interpretation letters have been completed.";
                        message3 = "The contact log has been updated.";
                        message4 = "These letters will be saved as part of the change history for this lab result.";
                        message5 = "A new contact log entry will be created.";
                    }
                }
                button.OnClientClick = "return DisplayWarningMessage('"
                    + message1 + "', '" + message2 + "', '" + message3 + "', '"
                    + message4 + "', '" + message5 + "', '" + message6 + "');";
            }
            else
            {
                button.Text = "Accept";
                button.OnClientClick = "return confirm('Please note that after pressing the Accept button this record will become static (non-editable). Are you sure that you want to accept this lab result?');";
            }
        }
    }
    
    /// <summary>
    /// Returns an array with all countries
    /// </summary>     
    public static StringCollection GetCountries()
    {
        StringCollection countries = new StringCollection();
        countries.AddRange(_countries);
        return countries;
    }
    /// <summary>
    /// Returns sorted list of all countries
    /// </summary>
    /// <param name="insertEmpty"></param>
    /// <returns></returns>
    public static SortedList GetCountries(bool insertEmpty)
    {
        SortedList countries = new SortedList();
        if (insertEmpty)
            countries.Add("", "Please select one...");
        foreach (String country in _countries)
            countries.Add(country, country);
        return countries;
    }
    /// <summary>
    /// highlights controls on selection 
    /// </summary>
    /// <param name="container"></param>
    /// <param name="className"></param>
    public static void SetInputControlsHighlight(Control container, string className)
    {
        foreach (Control ctl in container.Controls)
        {
            if ( ctl is TextBox || ctl is DropDownList ||
                ctl is ListBox || ctl is CheckBox || ctl is RadioButton ||
                ctl is RadioButtonList || ctl is CheckBoxList )
            {
                WebControl wctl = ctl as WebControl;
                if (wctl != null)
                {
                    wctl.Attributes.Add("onfocus", string.Format("this.className = '{0}';", className));
                    wctl.Attributes.Add("onblur", "this.className = '';");
                }                
            }
            else
            {
                if (ctl.Controls.Count > 0)
                    SetInputControlsHighlight(ctl, className);
            }
        }
    }
    //public static void SetUpHealthFactorCheckBoxes(Control container, int factorId)
    //{

    //    foreach (Control ctl in container.Controls)
    //    {
    //        if (ctl is CheckBoxList)
    //        {
    //                CheckBoxList boxList = ctl as CheckBoxList;

    //                ListItem item = boxList.Items.FindByValue(factorId.ToString());
    //                if (item != null)
    //                {
    //                    //if we find factor nothing else to do here 
    //                    item.Selected = true;
    //                    return;
    //                }   
    //        }
    //        else
    //        {
    //            if (ctl.Controls.Count > 0)
    //                SetUpHealthFactorCheckBoxes(ctl, factorId);
    //        }
    //    }    
    //}
    /// <summary>
    /// Utility to build a control list of pre-defined control type within a container(usually panel)
    /// </summary>
    /// <param name="container"></param>
    /// <param name="type"></param>
    /// <returns></returns>
    public static List<Control> BuildControlList(Control container, Type type) 
    {
        List<Control> controlList = new List<Control>();

        AddControlsToList(container.Controls, type, controlList);

        return controlList; 
    }
    private static void AddControlsToList(ControlCollection collection, Type type, List<Control> listToFill)
    {
         foreach (Control control in collection)
         {
            if (control.GetType().Equals(type))
            {

                listToFill.Add(control);
               

            }
            else
            {
                if (control.Controls.Count > 0 )
                    AddControlsToList(control.Controls, type, listToFill);

            }
        }   
    }
    public static Boolean isInteger(string valueToCheck)
    {
        int _tempInt; 
		
		try
			{
				_tempInt = Convert.ToInt32( valueToCheck );
                return true;
			}
			catch
			{
				return false;
			}

    }
    public static Boolean isString(string valueToCheck)
    {
        string _tempString; 
		
		try
			{
                _tempString = valueToCheck.ToString();
                return true;
			}
			catch
			{
				return false;
			}

    }
    public static string GetConfigEntry(string key)
    {
        string Val = string.Empty ;   
        if (WebConfigurationManager.AppSettings.Count > 0)
                Val = WebConfigurationManager.AppSettings.Get(key);

        return Val;
    }
    /// <summary>
    /// useful function to convert to various nullable types from string
    /// </summary>
    /// <typeparam name="T"></typeparam>
    /// <param name="input"></param>
    /// <returns></returns>
    public static T? ConvertStringToNullable<T>(string input) where T:struct
    {
        T? result = null;
        
        TypeConverter converter = TypeDescriptor.GetConverter(typeof(T));

        if (string.IsNullOrEmpty(input))
        {
            result = null;
        }
        else
        {
            try
            {
                result = (T)converter.ConvertFromString(input);
            }
            catch
            {
                result = null;
            }
           
        }

        return result;
    }
    public static Nullable<T> ToNullable<T>(this string s) where T : struct
    {
        Nullable<T> result = new Nullable<T>();
        try
        {
            if (!string.IsNullOrEmpty(s) && s.Trim().Length > 0)
            {
                TypeConverter conv = TypeDescriptor.GetConverter(typeof(T));
                result = (T)conv.ConvertFrom(s);
            }
        }
        catch { }
        return result;
    }
    public static int getIntFromString(string inString)
    {
        int returnInt = 0;
        int tempInt;
        bool result = int.TryParse(inString, out tempInt);
        if (result)
        {
            returnInt = tempInt;
        }
        return returnInt;
    }
    public static string getShortDateStringFromNullableDate(DateTime? inDate)
    {
        string returnString = "";
        if (inDate.HasValue)
        {
            returnString = Convert.ToDateTime(inDate).ToShortDateString();
        }
        return returnString;
    }

    public static string getCrsSetting(string name)
    {
        string returnString = "";

        RegistriesLinqDataContext dc = new RegistriesLinqDataContext();

        var settings = (from s in dc.CRS_SETTINGSs
                        where s.NAME == name
                        select s.VALUE).SingleOrDefault();

        if (settings == null)
            settings = "";

        returnString = settings.ToString();

        return returnString;
    }
    public static int GetDefaultTimeout()
    {
        return 600;
    }

    public static void RemoveAllLettersFromPrintQueue()
    {
        LabResultsIntrprtnManager manager = new LabResultsIntrprtnManager();
        LETTER_BATCH_PRINTING_STATUS batchPrintStatus = manager.GetBatchPrintingStatus();

        if (batchPrintStatus != null)
        {
            batchPrintStatus.SetAsChangeTrackingRoot();
            if (batchPrintStatus.PATIENT_FRAGMENT_BATCH_PRINTED
                && batchPrintStatus.PROVIDER_FRAGMENT_BATCH_PRINTED)
            {
                var letters = manager.GetAllInterpretationLettersInPrintQueue((int)EfrWorkCaseType.FragmentAnalysis);
                letters.ForEach(letter => manager.UpdateInterpretationLetterPrintQueue(letter, false));
                batchPrintStatus.PATIENT_FRAGMENT_BATCH_PRINTED = false;
                batchPrintStatus.PROVIDER_FRAGMENT_BATCH_PRINTED = false;
                batchPrintStatus.SetAsUpdateOnSubmit();
                HttpContext.Current.Session.Remove("PrintCompletedLetters");
            }

            if (batchPrintStatus.PROVIDER_BIO_BATCH_PRINTED
                && batchPrintStatus.PATIENT_BIO_BATCH_PRINTED)
            {
                var letters = manager.GetAllInterpretationLettersInPrintQueue((int)EfrWorkCaseType.BiologicalMonitoring);
                letters.ForEach(letter => manager.UpdateInterpretationLetterPrintQueue(letter, false));
                batchPrintStatus.PROVIDER_BIO_BATCH_PRINTED = false;
                batchPrintStatus.PATIENT_BIO_BATCH_PRINTED = false;
                batchPrintStatus.SetAsUpdateOnSubmit();
                HttpContext.Current.Session.Remove("PrintCompletedLetters");
            }

            if (batchPrintStatus.LINQEntityState == EntityState.Modified) 
            {
                manager.UpdateLetterBatchPrintingStatus(batchPrintStatus);
            }
        }
    }

    public static bool IsDestinationUrlLocal(string pDestinationUrl, string currentHost)
    {
        bool isValidUrl = false;
        string localHost = string.Concat("https://", currentHost, "/");
        string newUrl = string.Concat(localHost, pDestinationUrl);
        Uri destinationUrl;
        Uri.TryCreate(newUrl, UriKind.Absolute, out destinationUrl);
        if (destinationUrl != null && currentHost == destinationUrl.Host)
        {
            isValidUrl = true;
        }
        return isValidUrl;
    }
}
